<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User; // Assuming users are linked to showrooms
use App\Models\Showroom;
use App\Models\PersonalAccessToken; // Ensure you have this model imported for token management
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Str; // For generating random passwords if needed
use Carbon\Carbon; // For handling expiry dates

class ShowroomApiTokenController extends Controller
{
    // This method renders the view to generate tokens
    public function showGenerateForm()
    {
        // You might want to restrict this further, e.g., only for 'admin' role
        // $this->authorize('generate-api-tokens'); // If using Laravel policies

        $showrooms = Showroom::orderBy('name')->get();

        // Define common abilities a showroom token might need
        // These map to the 'abilities' you allow in your Sanctum token generation
        $availableAbilities = [
            'stock:submit' => 'Submit Daily Stock Data',
            // Add other abilities as your system grows, e.g.,
            // 'order:view' => 'View Orders for Showroom',
            // 'product:sync' => 'Sync Product Catalog',
        ];

        return view('admin.tokens.generate_showroom_token', compact('showrooms', 'availableAbilities'));
    }

    // This method handles the API request to generate the token
    public function generateToken(Request $request)
    {
        // IMPORTANT: Ensure only authorized users (e.g., admins) can call this API
        // Add authentication and authorization middleware to the route for this method.
        // Example: Route::post('/admin/tokens/generate', [...])->middleware(['auth:sanctum', 'role:admin']);

        $request->validate([
            'showroom_id' => 'required|exists:showrooms,id',
            'token_name' => 'required|string|max:255',
            'abilities' => 'nullable|array',
            'abilities.*' => 'string', // Validate each ability string
            'expires_at' => 'nullable|date|after_or_equal:today',
        ]);

        $showroom = Showroom::find($request->showroom_id);

        // Option 1: Find or create a dedicated API user for the showroom
        // This is generally recommended: a non-human user tied to the showroom
        $user = User::firstOrCreate(
            ['showroom_id' => $showroom->id, 'email' => "api_user_{$showroom->id}@yourdomain.com"],
            [
                'name' => "API User for {$showroom->name}",
                'password' => Hash::make(Str::random(32)), // Random password, not used for API token auth
                'email_verified_at' => now(), // Mark as verified if an API user
            ]
        );

        // Clean up old tokens for this user/name if you want to ensure only one active token per name
        // Or, consider deleting all tokens for this API user before creating a new one
        $user->tokens()->where('name', $request->token_name)->delete();

        // Generate the new Sanctum token
        $token = $user->createToken(
            $request->token_name,
            $request->abilities ?? ['*'], // Default to all abilities if none specified
            $request->expires_at ? Carbon::parse($request->expires_at) : null
        );

        return response()->json([
            'message' => 'API Token generated successfully.',
            'token' => $token->plainTextToken,
            'token_id' => $token->accessToken->id, // Useful for managing/revoking tokens
            'user_id' => $user->id,
            'showroom_name' => $showroom->name,
            'expires_at' => $token->accessToken->expires_at ? $token->accessToken->expires_at->toDateTimeString() : null,
        ]);
    }

    // Optionally, methods to revoke tokens by ID or showroom_id
    public function revokeToken(Request $request, $tokenId)
    {
        // Ensure only authorized users can revoke tokens
        // Example: $request->user()->tokenCan('admin:revoke-tokens');

        // $token = $request->user()->tokens()->where('id', $tokenId)->first(); // Revoke only current user's tokens
        // OR if an admin, allow revoking any token by ID:
        $token = PersonalAccessToken::find($tokenId); // Ensure PersonalAccessToken model is imported

        if (!$token) {
            return response()->json(['message' => 'Token not found or unauthorized.'], 404);
        }

        $token->delete();

        return response()->json(['message' => 'Token revoked successfully.']);
    }

    // You might want a list of generated tokens for management
    public function listShowroomTokensView(Request $request)
    {
        // Option 1 (Recommended): Fetch tokens based on a naming convention.
        // This is flexible if tokens are generated by admins but meant for various showroom entities.
        // Assumes token names like "Showroom ABC Token", "My Showroom Access"
        $showroomTokens = PersonalAccessToken::where('tokenable_type', 'App\\Models\\User') // Link to the User model
                                            
                                            // You might also filter by the admin user who created it,
                                            // if you store 'creator_id' or similar in a custom field or ability.
                                            ->orderBy('created_at', 'desc')
                                            ->paginate(20); // Added pagination for a better UX

        // Option 2 (Alternative - if tokens are exclusively generated by the current admin and tied to their user model for management)
        // $adminUser = $request->user();
        // $showroomTokens = $adminUser->tokens()
        //                            ->where('name', 'like', 'Showroom % Token') // Filter for relevant names
        //                            ->orderBy('created_at', 'desc')
        //                            ->paginate(10);


        return view('admin.tokens.index', compact('showroomTokens'));
    }

}