<?php

namespace App\Http\Controllers;

use App\Models\DiscountCampaign;
use App\Models\Product; // Assuming you have a Product model
use App\Models\Discount; // Assuming you have a Discount model
use Illuminate\Http\Request;

class DiscountCampaignController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        // Fetch all discount campaigns with pagination, 10 items per page
        // Eager load 'discount' relationship
        $discountCampaigns = DiscountCampaign::with(['discount'])->latest()->paginate(10);

        // Collect all product_ids from campaigns where discount_type is 'products'
        $allProductIds = [];
        foreach ($discountCampaigns as $campaign) {
            if ($campaign->discount_type === 'products' && !empty($campaign->product_ids)) {
                $allProductIds = array_merge($allProductIds, $campaign->product_ids);
            }
        }

        // Fetch all unique products in one go
        $products = collect();
        if (!empty($allProductIds)) {
            $products = Product::whereIn('id', array_unique($allProductIds))->get()->keyBy('id');
        }

        // Attach the related products to each campaign object
        foreach ($discountCampaigns as $campaign) {
            if ($campaign->discount_type === 'products' && !empty($campaign->product_ids)) {
                $campaignProducts = collect();
                foreach ($campaign->product_ids as $productId) {
                    if ($products->has($productId)) {
                        $campaignProducts->push($products->get($productId));
                    }
                }
                // Add a new attribute 'relatedProducts' to the campaign model instance
                $campaign->setRelation('relatedProducts', $campaignProducts);
            } else {
                $campaign->setRelation('relatedProducts', collect()); // Ensure it's always a collection
            }
        }

        return view('discount_campaigns.index', compact('discountCampaigns'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        // Fetch products and discounts for select dropdowns
        $products = Product::orderBy('productname')->get(['id', 'productname', 'productcategory']);
        $discounts = Discount::orderBy('name')->get(['id', 'name', 'value', 'type']);

        // Get unique product categories for the category selection
        $categories = Product::select('productcategory')->distinct()->pluck('productcategory')->filter()->sort()->values();

        return view('discount_campaigns.create', compact('products', 'discounts', 'categories'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // Validate the incoming request data
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'discount_id' => 'required|exists:discounts,id',
            'discount_type' => 'required|in:global,category,products',
            'category_name' => 'nullable|string|max:255',
            'product_ids' => 'nullable|array',
            'product_ids.*' => 'exists:products,id',
            'is_active' => 'boolean',
        ]);

        // Handle conditional validation based on discount_type
        if ($request->discount_type === 'category') {
            $request->validate(['category_name' => 'required|string|max:255']);
            $request->merge(['product_ids' => null]);
        } elseif ($request->discount_type === 'products') {
            $request->validate(['product_ids' => 'required|array|min:1']);
            $request->merge(['category_name' => null]);
        } else { // global
            $request->merge(['category_name' => null, 'product_ids' => null]);
        }

        // Create a new DiscountCampaign instance
        DiscountCampaign::create($request->all());

        return redirect()->route('discount_campaigns.index')
                         ->with('success', 'Discount campaign created successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\DiscountCampaign  $discountCampaign
     * @return \Illuminate\Http\Response
     */
    public function edit(DiscountCampaign $discountCampaign)
    {
        // Fetch products and discounts for select dropdowns
        $products = Product::orderBy('productname')->get(['id', 'productname', 'productcategory']);
        $discounts = Discount::orderBy('name')->get(['id', 'name', 'value', 'type']);

        // Get unique product categories for the category selection
        $categories = Product::select('productcategory')->distinct()->pluck('productcategory')->filter()->sort()->values();

        return view('discount_campaigns.edit', compact('discountCampaign', 'products', 'discounts', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\DiscountCampaign  $discountCampaign
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, DiscountCampaign $discountCampaign)
    {
        // Validate the incoming request data
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'discount_id' => 'required|exists:discounts,id',
            'discount_type' => 'required|in:global,category,products',
            'category_name' => 'nullable|string|max:255',
            'product_ids' => 'nullable|array',
            'product_ids.*' => 'exists:products,id',
            'is_active' => 'boolean',
        ]);

        // Handle conditional validation based on discount_type
        if ($request->discount_type === 'category') {
            $request->validate(['category_name' => 'required|string|max:255']);
            $request->merge(['product_ids' => null]);
        } elseif ($request->discount_type === 'products') {
            $request->validate(['product_ids' => 'required|array|min:1']);
            $request->merge(['category_name' => null]);
        } else { // global
            $request->merge(['category_name' => null, 'product_ids' => null]);
        }

        // Update the DiscountCampaign instance
        $discountCampaign->update($request->all());

        return redirect()->route('discount_campaigns.index')
                         ->with('success', 'Discount campaign updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\DiscountCampaign  $discountCampaign
     * @return \Illuminate\Http\Response
     */
    public function destroy(DiscountCampaign $discountCampaign)
    {
        // Delete the DiscountCampaign instance
        $discountCampaign->delete();

        return redirect()->route('discount_campaigns.index')
                         ->with('success', 'Discount campaign deleted successfully.');
    }
}
