<?php

namespace App\Http\Controllers;

use App\Models\Permission;
use App\Models\Role; // Assuming you have a Role model for the relationship
use Illuminate\Http\Request;

class PermissionController extends Controller
{
    public function index()
    {
        $permissions = Permission::paginate(10); // Adjust pagination as needed
        return view('permissions.index', compact('permissions'));
    }

    public function create()
    {
        $roles = Role::all(); // Fetch all roles for the select box
        return view('permissions.create', compact('roles'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:permissions,name',
            'guard_name' => 'required|string|max:255',
            'roles' => 'nullable|array', // For attaching roles
        ]);

        $permission = Permission::create($request->only(['name', 'guard_name']));
        if ($request->has('roles')) {
            $permission->roles()->sync($request->input('roles'));
        }

        return redirect()->route('permissions.index')->with('success', 'Permission created successfully.');
    }

    public function edit(Permission $permission)
    {
        $roles = Role::all(); // Fetch all roles for the select box
        $assignedRoles = $permission->roles->pluck('id')->toArray(); // Get IDs of roles already assigned
        return view('permissions.edit', compact('permission', 'roles', 'assignedRoles'));
    }

    public function update(Request $request, Permission $permission)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:permissions,name,' . $permission->id,
            'guard_name' => 'required|string|max:255',
            'roles' => 'nullable|array', // For attaching roles
        ]);

        $permission->update($request->only(['name', 'guard_name']));
        if ($request->has('roles')) {
            $permission->roles()->sync($request->input('roles'));
        } else {
            $permission->roles()->detach(); // Detach all if no roles selected
        }

        return redirect()->route('permissions.index')->with('success', 'Permission updated successfully.');
    }

    public function destroy(Permission $permission)
    {
        $permission->delete();
        return redirect()->route('permissions.index')->with('success', 'Permission deleted successfully.');
    }
}