<?php

namespace App\Http\Controllers;

use App\Models\Showroom;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ShowroomController extends Controller
{
    public function index(Request $request)
    {
        $query = Showroom::query();
        
        // Search functionality
        if ($request->has('search') && $request->search != '') {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'LIKE', "%{$search}%")
                  ->orWhere('code', 'LIKE', "%{$search}%")
                  ->orWhere('city', 'LIKE', "%{$search}%")
                  ->orWhere('manager_name', 'LIKE', "%{$search}%");
            });
        }
        
        // Status filter
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }
        
        $showrooms = $query->orderBy('created_at', 'desc')->paginate(15);
        
        return view('sr.showroom.index', compact('showrooms'));
    }

    public function create()
    {
        return view('sr.showroom.create');
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|unique:showrooms,code',
            'address' => 'nullable|string',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:100',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'manager_name' => 'nullable|string|max:255',
            'manager_phone' => 'nullable|string|max:20',
            'manager_email' => 'nullable|email|max:255',
            'status' => 'required|in:active,inactive',
            'opening_time' => 'nullable|date_format:H:i',
            'closing_time' => 'nullable|date_format:H:i',
            'description' => 'nullable|string'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        Showroom::create($request->all());

        return redirect()->route('sr.showroom.index')
            ->with('success', 'Showroom created successfully!');
    }

    public function edit(Showroom $showroom)
    {
        return view('sr.showroom.edit', compact('showroom'));
    }

    public function update(Request $request, Showroom $showroom)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|unique:showrooms,code,' . $showroom->id,
            'address' => 'nullable|string',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:100',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'manager_name' => 'nullable|string|max:255',
            'manager_phone' => 'nullable|string|max:20',
            'manager_email' => 'nullable|email|max:255',
            'status' => 'required|in:active,inactive',
            'opening_time' => 'nullable|date_format:H:i',
            'closing_time' => 'nullable|date_format:H:i',
            'description' => 'nullable|string'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $showroom->update($request->all());

        return redirect()->route('sr.showroom.index')
            ->with('success', 'Showroom updated successfully!');
    }

    public function destroy(Showroom $showroom)
    {
        try {
            $showroom->delete();
            return redirect()->route('sr.showroom.index')
                ->with('success', 'Showroom deleted successfully!');
        } catch (\Exception $e) {
            return redirect()->route('sr.showroom.index')
                ->with('error', 'Error deleting showroom: ' . $e->getMessage());
        }
    }
}