<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use App\Models\Product;
use App\Models\ProductStock;
use App\Models\Showroom; 
use App\Models\ProductPrice; // IMPORTANT: Import ProductPrice model!
use Illuminate\Support\Facades\Log;

class StockController extends Controller
{
    public function index(Request $request)
    {
        $stockDate = $request->input('date') ?? now()->format('Y-m-d');
        $showrooms = Showroom::all(); 

        // For the initial page load, we only need to fetch all products grouped by category
        // if the frontend will then use AJAX (via fetch method) to populate the actual stock data.
        // We remove the 'stocks' eager loading here as it's handled by the 'fetch' AJAX call.
        $products = Product::orderBy('productcategory')->get()
          ->groupBy('productcategory');

        return view('sr.productstock.index', compact('products', 'stockDate', 'showrooms'));
    }

    public function table()
    {
        $showrooms = Showroom::all(); 
        return view('sr.productstock.table', compact('showrooms'));
    }

    public function update(Request $request)
    {
        $data = $request->validate([
            'stock_date' => 'required|date',
            'showroom_id' => 'required|exists:showrooms,id', 
            'entries' => 'required|array'
        ]);

        foreach ($data['entries'] as $productId => $fields) {
            ProductStock::updateOrCreate(
                [
                    'product_id' => $productId,
                    'stock_date' => $data['stock_date'],
                    'showroom_id' => $data['showroom_id'] 
                ],
                [
                    'opening_stock' => $fields['opening_stock'],
                    'received' => $fields['received'],
                    'return_damage' => $fields['return_damage'],
                    'closing_stock' => $fields['closing_stock']
                ]
            );
        }

        return response()->json(['message' => 'Stock saved successfully']);
    }

    public function fetch(Request $request)
    {
        $date = Carbon::parse($request->input('date', today()));
        $prevDate = $date->copy()->subDay();
        $selectedShowroomId = $request->input('showroom_id'); 

        $request->validate([
            'showroom_id' => 'nullable|exists:showrooms,id',
        ]);

        // 1. Fetch all today's stock data for the selected showroom
        $todayStocksQuery = ProductStock::whereDate('stock_date', $date->toDateString())
            ->with('product', 'showroom'); 
        
        if ($selectedShowroomId) {
            $todayStocksQuery->where('showroom_id', $selectedShowroomId);
        }
        $todayStocks = $todayStocksQuery->get()->keyBy('product_id');

        // 2. Fetch all previous day's closing stocks for the selected showroom
        $prevDayStocksQuery = ProductStock::whereDate('stock_date', $prevDate->toDateString());
        if ($selectedShowroomId) {
            $prevDayStocksQuery->where('showroom_id', $selectedShowroomId);
        }
        $prevDayStocks = $prevDayStocksQuery->get()->keyBy('product_id');

        // 3. Get all products to ensure all products are listed, even if no stock entries
        $allProducts = Product::orderBy('productcategory')->get();

        // 4. Determine effective prices for all products for the given date and showroom
        // This is the core change: fetching prices from ProductPrice
        $relevantPrices = ProductPrice::where('effective_start_date', '<=', $date->toDateString())
            ->where(function ($query) use ($date) {
                $query->whereNull('effective_end_date')
                      ->orWhere('effective_end_date', '>=', $date->toDateString());
            })
            ->where(function ($query) use ($selectedShowroomId) {
                // Prices that are not showroom overrides (general prices) and have null showroom_id
                $query->where(function ($q) {
                    $q->where('is_showroom_override', false)
                      ->whereNull('showroom_id');
                });

                // OR prices that are showroom overrides for the selected showroom
                if ($selectedShowroomId) {
                    $query->orWhere(function ($q) use ($selectedShowroomId) {
                        $q->where('is_showroom_override', true)
                          ->where('showroom_id', $selectedShowroomId);
                    });
                }
            })
            ->get();

        // Build a map of product_id => effective_price
        // Prioritizes showroom-specific prices over general prices, and latest start date if multiple
        $effectivePricesMap = $relevantPrices->groupBy('product_id')->mapWithKeys(function ($pricesForProduct, $productId) use ($selectedShowroomId) {
            // Filter for showroom-specific prices for the selected showroom first
            $showroomPrices = $pricesForProduct->filter(function ($price) use ($selectedShowroomId) {
                return $price->is_showroom_override && $price->showroom_id == $selectedShowroomId;
            });

            if ($showroomPrices->isNotEmpty()) {
                // If showroom-specific prices exist, pick the one with the latest effective_start_date
                return [$productId => $showroomPrices->sortByDesc('effective_start_date')->first()->price];
            }

            // If no showroom-specific prices, fall back to general prices (is_showroom_override = false, showroom_id = null)
            $generalPrices = $pricesForProduct->filter(function ($price) {
                return !$price->is_showroom_override && is_null($price->showroom_id);
            });

            if ($generalPrices->isNotEmpty()) {
                // Pick the general price with the latest effective_start_date
                return [$productId => $generalPrices->sortByDesc('effective_start_date')->first()->price];
            }

            // If no relevant price found in ProductPrice, default to 0 for calculations
            return [$productId => 0]; 
        });

        $data = $allProducts->groupBy('productcategory')->map(function($group) use ($todayStocks, $prevDayStocks, $effectivePricesMap) {
            return $group->map(function($product) use ($todayStocks, $prevDayStocks, $effectivePricesMap) {
                $todayStock = $todayStocks->get($product->id); 
                $prevStock = $prevDayStocks->get($product->id);

                // Determine opening stock as per your existing logic
                $openingStock = $todayStock ? $todayStock->opening_stock : 
                                             ($prevStock ? $prevStock->closing_stock : 0);

                if ($todayStock && $todayStock->opening_stock == 0 && $prevStock) {
                    $openingStock = $prevStock->closing_stock;
                }

                $received = $todayStock ? $todayStock->received : 0;
                $returnDamage = $todayStock ? $todayStock->return_damage : 0;
                $closingStock = $todayStock ? $todayStock->closing_stock : 0;
                $sale = $openingStock + $received - $returnDamage - $closingStock;

                // Get the determined effective price for this product
                $productPrice = $effectivePricesMap->get($product->id) ?? 0;

                return [
                    'id' => $product->id,
                    'name' => $product->productname,
                    'price' => $productPrice, // Use the dynamically determined price from ProductPrice
                    'os' => $openingStock,
                    'received' => $received,
                    'r_d' => $returnDamage,
                    'cs' => $closingStock,
                    'sale' => $sale,
                    'money' => [
                        'os' => $openingStock * $productPrice,
                        'received' => $received * $productPrice,
                        'r_d' => $returnDamage * $productPrice,
                        'cs' => $closingStock * $productPrice,
                        'sale' => $sale * $productPrice
                    ],
                    'has_stock' => (bool)$todayStock,
                    'has_prev_stock' => (bool)$prevStock,
                    'showroom' => $todayStock && $todayStock->showroom ? [
                        'id' => $todayStock->showroom->id,
                        'name' => $todayStock->showroom->name, 
                    ] : null,
                ];
            });
        });

        $categorySums = $data->map(function ($products) {
            return $products->reduce(function ($sum, $item) {
                foreach (['os', 'received', 'r_d', 'cs', 'sale'] as $k) {
                    $sum[$k] += $item['money'][$k] ?? 0;
                }
                return $sum;
            }, ['os' => 0, 'received' => 0, 'r_d' => 0, 'cs' => 0, 'sale' => 0]);
        });

        Log::debug('Stock data:', [
            'date' => $date->toDateString(),
            'prevDate' => $prevDate->toDateString(),
            'selected_showroom_id' => $selectedShowroomId,
            'products_count' => $allProducts->count(),
            'today_stocks_count' => $todayStocks->count(),
            'prev_stocks_count' => $prevDayStocks->count(),
            'effective_prices_count' => $effectivePricesMap->count(),
            'sample_product' => $allProducts->first() ? $allProducts->first()->toArray() : null,
            'sample_today_stock' => $todayStocks->first() ? $todayStocks->first()->toArray() : null,
            'sample_prev_stock' => $prevDayStocks->first() ? $prevDayStocks->first()->toArray() : null,
            'sample_effective_price' => $effectivePricesMap->first()
        ]);

        return response()->json([
            'date' => $date->toDateString(),
            'data' => $data,
            'categorySums' => $categorySums,
            'prevDate' => $prevDate->toDateString(),
            'selectedShowroomId' => $selectedShowroomId 
        ]);
    }
}