@extends('layouts.main')

@section('content')
<div class="container-fluid">
    <div class="d-flex justify-content-between mb-2 align-items-center">
        <h6 class="fw-bold small">Stock Entry & Summary</h6>
        <input type="date" id="stockDate" value="{{ date('Y-m-d') }}" class="form-control form-control-sm w-auto">
    </div>

    <div class="col-md-4 mb-3">
        <label for="showroomSelect">Select Showroom:</label>
        <select id="showroomSelect" class="form-control">
            <option value="">All Showrooms</option>
            @foreach($showrooms as $showroom)
                <option value="{{ $showroom->id }}">{{ $showroom->name }}</option>
            @endforeach
        </select>
    </div>

    {{-- Main Summary Table --}}
    <table class="table table-sm table-bordered text-end mb-2" id="summaryTable">
        <thead class="table-light">
            <tr>
                <th>OS ৳</th><th>Received ৳</th><th>R/D ৳</th><th>CS ৳</th><th>Sale ৳</th>
            </tr>
        </thead>
        <tbody><tr><td colspan="5">Loading...</td></tr></tbody>
    </table>

    {{-- NEW: Accessories Summary Table Area --}}
    <div id="accessoriesSummaryTableArea">
        {{-- Content will be injected by JavaScript --}}
    </div>

    {{-- NEW: Out Accessories Summary Table Area --}}
    <div id="outAccessoriesSummaryTableArea">
        {{-- Content will be injected by JavaScript --}}
        
    </div>
                <div class="text-end mt-2">
        <button class="btn btn-sm btn-primary" id="saveBtn">Save</button>
    </div>

    <div class="d-flex justify-content-between">

        <div id="stockTableWrapper" style="width: 60%;">
            <h6 class="text-muted small mb-1">Stock Quantity</h6>
            <div class="accordion" id="categoryAccordion">
                {{-- Category accordions will be injected here by JavaScript --}}
            </div>
        </div>

        <div id="categorySummaryWrapper" style="width: 39%;">
            <h6 class="text-muted small mb-1">Category-wise Summary</h6>
            <div id="categorySummaryContainer">
                {{-- Category summaries will be injected here by JavaScript --}}
            </div>
        </div>
    </div>


</div>

@endsection



@push('scripts')
<style>
    table {
        font-size: 11px;
    }
    td, th {
        padding: 2px !important;
        vertical-align: middle;
    }
    input[type="number"]::-webkit-inner-spin-button,
    input[type="number"]::-webkit-outer-spin-button {
        -webkit-appearance: none;
        margin: 0;
    }
    input[type="number"] {
        appearance: textfield;
        -moz-appearance: textfield;
    }
    input.form-control-xs {
        padding: 1px 4px;
        height: 24px;
        font-size: 11px;
        line-height: 1.2;
    }
    .accordion-button:not(.collapsed) {
        background-color: #f8f9fa;
    }
    .accordion-button {
        padding: 0.25rem 0.5rem;
        font-size: 11px;
    }
    .accordion-body {
        padding: 0;
    }
</style>

<script>

    let stockData = {}, currentDate = document.getElementById("stockDate").value;

    document.addEventListener('DOMContentLoaded', () => {
        // Initial fetch on page load
        fetchData(currentDate);

        // Event listener for date change
        document.getElementById("stockDate").addEventListener("change", e => {
            currentDate = e.target.value; // Update global currentDate
            fetchData(currentDate);
        });

        // Event listener for showroom selection change (NEW)
        document.getElementById("showroomSelect").addEventListener("change", () => {
            fetchData(currentDate); // Re-fetch data for the current date and new showroom
        });

        // Event listener for save button
        document.getElementById("saveBtn").addEventListener("click", saveData);
    });

    // fetchData function (UNTOUCHED)
    function fetchData(date) {
        const showroomId = document.getElementById("showroomSelect").value; // Get showroomId here
        let url = `{{ route('stocks.fetch') }}?date=${date}`;
        if (showroomId) {
            url += `&showroom_id=${showroomId}`;
        }

        fetch(url)
            .then(res => {
                if (!res.ok) {
                    return res.json().then(err => { throw err; });
                }
                return res.json();
            })
            .then(({ date, data, prevDate, selectedShowroomId }) => {
                currentDate = date;
                stockData = data;

                document.getElementById("stockDate").value = date;
                document.getElementById("showroomSelect").value = selectedShowroomId || '';

                renderTables(); // renderTables now triggers the initial deduction

                console.log(`Loaded data for ${date} (Showroom: ${selectedShowroomId || 'All'}). Previous date: ${prevDate}`);
            })
            .catch(error => {
                console.error('Error fetching stock data:', error);
                alert('Failed to load stock data. Check console for details.');
                document.getElementById('categoryAccordion').innerHTML = '<p class="text-danger p-3">Failed to load stock data.</p>';
                document.getElementById("accessoriesSummaryTableArea").innerHTML = '';
                document.getElementById("outAccessoriesSummaryTableArea").innerHTML = '';
                document.querySelector("#summaryTable tbody").innerHTML = '<tr><td colspan="5">Error loading data.</td></tr>';
                document.getElementById("categorySummaryContainer").innerHTML = '';
            });
    }

    // renderTables function (MODIFIED)
    function renderTables() {
        const accordion = document.getElementById('categoryAccordion');
        accordion.innerHTML = '';

        // Define categories where OS, Received, R/D should be readonly and Sale = CS
        const specialCategories = ['Accessories', 'Out Accessories', 'Soft Drinks & Juices'];

        const accessoriesSummaryArea = document.getElementById("accessoriesSummaryTableArea");
        if (accessoriesSummaryArea) {
            accessoriesSummaryArea.innerHTML = `
                    <h5 class="mt-3">Accessories Summary</h5>
                    <table class="table table-bordered table-sm text-end mb-3">
                        <thead class="table-dark">
                            <tr>
                                <th>OS ৳</th><th>Received ৳</th><th>R/D ৳</th><th>CS ৳</th><th>Sale ৳</th>
                            </tr>
                        </thead>
                        <tbody id="accessoriesSummaryTableBody"></tbody>
                    </table>
                `;
        }

        const outAccessoriesSummaryArea = document.getElementById("outAccessoriesSummaryTableArea");
        if (outAccessoriesSummaryArea) {
            outAccessoriesSummaryArea.innerHTML = `
                    <h5 class="mt-3">Out Accessories Summary</h5>
                    <table class="table table-bordered table-sm text-end mb-3">
                        <thead class="table-dark">
                            <tr>
                                <th>OS ৳</th><th>Received ৳</th><th>R/D ৳</th><th>CS ৳</th><th>Sale ৳</th>
                            </tr>
                        </thead>
                        <tbody id="outAccessoriesSummaryTableBody"></tbody>
                    </table>
                `;
        }

        Object.entries(stockData).forEach(([cat, items], index) => {
            const accordionItem = document.createElement('div');
            accordionItem.className = 'accordion-item';
            accordionItem.innerHTML = `
                    <h2 class="accordion-header" id="heading${index}">
                        <button class="accordion-button collapsed" type="button"
                                data-bs-toggle="collapse" data-bs-target="#collapse${index}"
                                aria-expanded="false" aria-controls="collapse${index}">
                            ${cat}
                        </button>
                    </h2>
                    <div id="collapse${index}" class="accordion-collapse collapse"
                        aria-labelledby="heading${index}" data-bs-parent="#categoryAccordion">
                        <div class="accordion-body p-0">
                            <table class="table table-sm mb-0">
                                <thead>
                                    <tr>
                                        <th>Product</th>
                                        <th>Price</th>
                                        <th>OS</th>
                                        <th>Received</th>
                                        <th>R/D</th>
                                        <th>CS</th>
                                        <th>Sale</th>
                                    </tr>
                                </thead>
                                <tbody id="tbody${index}"></tbody>
                            </table>
                        </div>
                    </div>
                `;
            accordion.appendChild(accordionItem);

            const tbody = document.getElementById(`tbody${index}`);
            items.forEach((item) => {
                // Check if the current category is one of the special categories
                const isSpecialCategory = specialCategories.includes(cat);

                // Determine readonly state for OS, Received, R/D
                const osReadonly = (item.has_stock || item.has_prev_stock || isSpecialCategory) ? 'readonly' : '';
                const receivedReadonly = isSpecialCategory ? 'readonly' : '';
                const rdReadonly = isSpecialCategory ? 'readonly' : '';

                // Determine background color for readonly fields
                const readonlyBg = 'style="background-color:#e9ecef;"';

                // *** MODIFICATION START ***
                // Calculate 'sale' based on the client-side logic for initial render
                let displaySale;
                if (isSpecialCategory) {
                    displaySale = item.cs;
                } else {
                    displaySale = (item.os || 0) + (item.received || 0) - (item.r_d || 0) - (item.cs || 0);
                }
                // *** MODIFICATION END ***

                const row = document.createElement('tr');
                row.dataset.id = item.id;
                row.dataset.cat = cat; // Ensure the category is stored in the dataset
                row.dataset.price = item.price;
                row.innerHTML = `
                        <td>${item.name}</td>
                        <td>${item.price.toFixed(2)}</td>
                        <td><input type="number" value="${item.os}" class="form-control form-control-xs" data-col="os" ${osReadonly} ${osReadonly ? readonlyBg : ''}></td>
                        <td><input type="number" value="${item.received}" class="form-control form-control-xs" data-col="received" ${receivedReadonly} ${receivedReadonly ? readonlyBg : ''}></td>
                        <td><input type="number" value="${item.r_d}" class="form-control form-control-xs" data-col="r_d" ${rdReadonly} ${rdReadonly ? readonlyBg : ''}></td>
                        <td><input type="number" value="${item.cs}" class="form-control form-control-xs" data-col="cs"></td>
                        <td><input type="number" value="${displaySale}" class="form-control form-control-xs" disabled style="background-color:#f8f9fa;"></td>
                    `;
                tbody.appendChild(row);
            });
        });

        if (typeof bootstrap !== 'undefined') {
            new bootstrap.Collapse(document.getElementById('categoryAccordion'), {
                toggle: false
            });
        }

        attachListeners();
        applyCrossProductDeductions();
        updateAccessoriesSummaryTable();
        updateOutAccessoriesSummaryTable();
        updateCategorySummary();
        updateSummaryTable();
    }

    // attachListeners function (UNTOUCHED)
    function attachListeners() {
        document.querySelectorAll('.accordion-body input[data-col]').forEach(input => {
            input.removeEventListener("input", updateRowAndSummaries);
            input.addEventListener("input", updateRowAndSummaries);
        });
    }

    // updateRowAndSummaries function (UNTOUCHED in its specialCategories definition)
    function updateRowAndSummaries() {
        const row = this.closest("tr");
        const category = row.dataset.cat;
        const specialCategories = ['Accessories', 'Out Accessories', 'Soft Drinks & Juices'];

        const price = parseFloat(row.dataset.price || 0);
        const os = +row.querySelector("input[data-col='os']").value || 0;
        const r = +row.querySelector("input[data-col='received']").value || 0;
        const rd = +row.querySelector("input[data-col='r_d']").value || 0;
        const cs = +row.querySelector("input[data-col='cs']").value || 0;

        let sale;

        if (specialCategories.includes(category)) {
            sale = cs;
        } else {
            sale = os + r - rd - cs;
        }

        row.querySelector("input[disabled]").value = sale;

        applyCrossProductDeductions();

        updateAccessoriesSummaryTable();
        updateOutAccessoriesSummaryTable();
        updateCategorySummary();
        updateSummaryTable();
    }

    // updateAccessoriesSummaryTable (UNTOUCHED)
    function updateAccessoriesSummaryTable() {
        let sum = {os:0, received:0, r_d:0, cs:0, sale:0};
        const targetCategory = 'Accessories';

        document.querySelectorAll(`tr[data-cat="${targetCategory}"]`).forEach(row => {
            const price = parseFloat(row.dataset.price || 0);
            const os = +row.querySelector("input[data-col='os']").value || 0;
            const r = +row.querySelector("input[data-col='received']").value || 0;
            const rd = +row.querySelector("input[data-col='r_d']").value || 0;
            const cs = +row.querySelector("input[data-col='cs']").value || 0;
            const sale = +row.querySelector("input[disabled]").value || 0;

            sum.os += os * price;
            sum.received += r * price;
            sum.r_d += rd * price;
            sum.cs += cs * price;
            sum.sale += sale * price;
        });

        const tbody = document.getElementById("accessoriesSummaryTableBody");
        if (tbody) {
            tbody.innerHTML = `
                    <tr>
                        <td class="text-end">${sum.os.toFixed(2)}</td>
                        <td class="text-end">${sum.received.toFixed(2)}</td>
                        <td class="text-end">${sum.r_d.toFixed(2)}</td>
                        <td class="text-end">${sum.cs.toFixed(2)}</td>
                        <td class="text-end fw-bold">${sum.sale.toFixed(2)}</td>
                    </tr>`;
        }
    }

    // updateOutAccessoriesSummaryTable (UNTOUCHED)
    function updateOutAccessoriesSummaryTable() {
        let sum = {os:0, received:0, r_d:0, cs:0, sale:0};
        const targetCategory = 'Out Accessories';

        document.querySelectorAll(`tr[data-cat="${targetCategory}"]`).forEach(row => {
            const price = parseFloat(row.dataset.price || 0);
            const os = +row.querySelector("input[data-col='os']").value || 0;
            const r = +row.querySelector("input[data-col='received']").value || 0;
            const rd = +row.querySelector("input[data-col='r_d']").value || 0;
            const cs = +row.querySelector("input[data-col='cs']").value || 0;
            const sale = +row.querySelector("input[disabled]").value || 0;

            sum.os += os * price;
            sum.received += r * price;
            sum.r_d += rd * price;
            sum.cs += cs * price;
            sum.sale += sale * price;
        });

        const tbody = document.getElementById("outAccessoriesSummaryTableBody");
        if (tbody) {
            tbody.innerHTML = `
                    <tr>
                        <td class="text-end">${sum.os.toFixed(2)}</td>
                        <td class="text-end">${sum.received.toFixed(2)}</td>
                        <td class="text-end">${sum.r_d.toFixed(2)}</td>
                        <td class="text-end">${sum.cs.toFixed(2)}</td>
                        <td class="text-end fw-bold">${sum.sale.toFixed(2)}</td>
                    </tr>`;
        }
    }

    // applyCrossProductDeductions function (MODIFIED in previous response, kept here for completeness)
    function applyCrossProductDeductions() {
        const specialCategories = ['Accessories', 'Out Accessories', 'Soft Drinks & Juices'];

        // Get the 'received' value for Shor Doi (ID 315)
        const shorDoiReceivedInput = document.querySelector(`tr[data-id="315"] input[data-col="received"]`);
        const shorDoiReceived = shorDoiReceivedInput ? +shorDoiReceivedInput.value || 0 : 0;

        // Get the 'received' value for Tok Doi (ID 316)
        const tokDoiReceivedInput = document.querySelector(`tr[data-id="316"] input[data-col="received"]`);
        const tokDoiReceived = tokDoiReceivedInput ? +tokDoiReceivedInput.value || 0 : 0;

        // --- Process Product ID 295 (affected by Shor Doi ID 315) ---
        const product295Row = document.querySelector(`tr[data-id="295"]`);
        if (product295Row) {
            const cat295 = product295Row.dataset.cat;
            const os295 = +product295Row.querySelector("input[data-col='os']").value || 0;
            const r295 = +product295Row.querySelector("input[data-col='received']").value || 0;
            const rd295 = +product295Row.querySelector("input[data-col='r_d']").value || 0;
            const cs295 = +product295Row.querySelector("input[data-col='cs']").value || 0;

            let sale295;
            if (specialCategories.includes(cat295)) {
                sale295 = cs295; // Use CS for special categories
            } else {
                sale295 = os295 + r295 - rd295 - cs295; // Standard calculation
            }
            let adjustedSale295 = sale295 - shorDoiReceived;
            product295Row.querySelector("input[disabled]").value = adjustedSale295;
        }

        // --- Process Product ID 296 (affected by Tok Doi ID 316) ---
        const product296Row = document.querySelector(`tr[data-id="296"]`);
        if (product296Row) {
            const cat296 = product296Row.dataset.cat;
            const os296 = +product296Row.querySelector("input[data-col='os']").value || 0;
            const r296 = +product296Row.querySelector("input[data-col='received']").value || 0;
            const rd296 = +product296Row.querySelector("input[data-col='r_d']").value || 0;
            const cs296 = +product296Row.querySelector("input[data-col='cs']").value || 0;

            let sale296;
            if (specialCategories.includes(cat296)) {
                sale296 = cs296; // Use CS for special categories
            } else {
                sale296 = os296 + r296 - rd296 - cs296; // Standard calculation
            }
            let adjustedSale296 = sale296 - tokDoiReceived;
            product296Row.querySelector("input[disabled]").value = adjustedSale296;
        }
    }


    // updateSummaryTable function (UNTOUCHED)
    function updateSummaryTable() {
        let sum = {os:0, received:0, r_d:0, cs:0, sale:0};

        // Define the categories to be completely excluded from ALL sums in the main summary table
        // 'Soft Drinks & Juices' IS REMOVED from this list to include it in the main summary.
        const categoriesToExcludeFromMainSummary = ['Accessories', 'Out Accessories', 'Ingredients'];

        document.querySelectorAll('.accordion-body tbody tr[data-id]').forEach(row => {
            const category = row.dataset.cat;
            const price = parseFloat(row.dataset.price || 0);
            const os = +row.querySelector("input[data-col='os']").value || 0;
            const r = +row.querySelector("input[data-col='received']").value || 0;
            const rd = +row.querySelector("input[data-col='r_d']").value || 0;
            const cs = +row.querySelector("input[data-col='cs']").value || 0;
            const sale = +row.querySelector("input[disabled]").value || 0;

            // ONLY add to the main summary if the category is NOT in the exclusion list
            if (!categoriesToExcludeFromMainSummary.includes(category)) {
                sum.os += os * price;
                sum.received += r * price;
                sum.r_d += rd * price;
                sum.cs += cs * price;
                sum.sale += sale * price;
            }
        });

        document.querySelector("#summaryTable tbody").innerHTML = `
                <tr>
                    <td class="text-end">${sum.os.toFixed(2)}</td>
                    <td class="text-end">${sum.received.toFixed(2)}</td>
                    <td class="text-end">${sum.r_d.toFixed(2)}</td>
                    <td class="text-end">${sum.cs.toFixed(2)}</td>
                    <td class="text-end fw-bold">${sum.sale.toFixed(2)}</td>
                </tr>`;
    }

    // updateCategorySummary function (UNTOUCHED)
    function updateCategorySummary() {
        const catTotals = {};

        // Categories that should NOT appear in this right-side category summary section
        // 'Soft Drinks & Juices' IS REMOVED from this list to make it appear in the right-side summary.
        const categoriesToExcludeFromRightSideSummary = ['Accessories', 'Out Accessories'];
        const excludedCategoriesFromSaleCalculation = ['Ingredients'];

        document.querySelectorAll('.accordion-body tbody tr[data-id]').forEach(row => {
            const cat = row.dataset.cat;
            const price = parseFloat(row.dataset.price || 0);
            const os = +row.querySelector("input[data-col='os']").value || 0;
            const r = +row.querySelector("input[data-col='received']").value || 0;
            const rd = +row.querySelector("input[data-col='r_d']").value || 0;
            const cs = +row.querySelector("input[data-col='cs']").value || 0;

            const sale = +row.querySelector("input[disabled]").value || 0;

            // Process all categories EXCEPT those specifically excluded from the right-side summary.
            if (!categoriesToExcludeFromRightSideSummary.includes(cat)) {
                catTotals[cat] ??= { os: 0, received: 0, r_d: 0, cs: 0, sale: 0 };
                catTotals[cat].os += os * price;
                catTotals[cat].received += r * price;
                catTotals[cat].r_d += rd * price;
                catTotals[cat].cs += cs * price;

                // Exclude sales for 'Ingredients' from its own category summary, but include others.
                if (!excludedCategoriesFromSaleCalculation.includes(cat)) {
                    catTotals[cat].sale += sale * price;
                } else {
                    catTotals[cat].sale += 0;
                }
            }
        });

        let html = '';
        Object.entries(catTotals).forEach(([cat, s]) => {
            html += `
                <div class="border p-2 mb-2 bg-light">
                    <div class="fw-bold small mb-1">${cat}</div>
                    <table class="table table-bordered table-sm text-end mb-0" style="font-size:11px;">
                        <thead class="table-light">
                            <tr>
                                <th>OS ৳</th><th>Received ৳</th><th>R/D ৳</th><th>CS ৳</th><th>Sale ৳</th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <td>${s.os.toFixed(2)}</td>
                                <td>${s.received.toFixed(2)}</td>
                                <td>${s.r_d.toFixed(2)}</td>
                                <td>${s.cs.toFixed(2)}</td>
                                <td><strong>${s.sale.toFixed(2)}</strong></td>
                            </tr>
                        </tbody>
                    </table>
                </div>`;
        });

        document.getElementById("categorySummaryContainer").innerHTML = html;
    }

    // saveData function (UNTOUCHED)
    function saveData() {
        const stockDate = document.getElementById("stockDate").value;
        const showroomId = document.getElementById("showroomSelect").value;
        const payload = {};

        document.querySelectorAll('.accordion-body tbody tr[data-id]').forEach(row => {
            const id = row.dataset.id;
            payload[id] = {
                opening_stock: +row.querySelector("input[data-col='os']").value || 0,
                received: +row.querySelector("input[data-col='received']").value || 0,
                return_damage: +row.querySelector("input[data-col='r_d']").value || 0,
                closing_stock: +row.querySelector("input[data-col='cs']").value || 0
            };
        });

        console.log('Saving data:', { stock_date: stockDate, showroom_id: showroomId, entries: payload });

        fetch("{{ route('stocks.update') }}", {
            method: "POST",
            headers: {
                'X-CSRF-TOKEN': "{{ csrf_token() }}",
                'Content-Type': "application/json",
                'Accept': 'application/json'
            },
            body: JSON.stringify({
                stock_date: stockDate,
                showroom_id: showroomId,
                entries: payload
            })
        })
            .then(res => {
                if (!res.ok) {
                    return res.json().then(err => { throw err; });
                }
                return res.json();
            })
            .then(data => {
                console.log('Save response:', data);
                alert(data.message || 'Stock saved successfully');
                fetchData(document.getElementById("stockDate").value);
            })
            .catch(error => {
                console.error('Save error:', error);
                alert(error.message || 'Failed to save stock. Check console for details.');
            });
    }

</script>


@endpush